/*
 * 8x8 dot matrix led(BU5004R and compatible) FW (Ver 1.00)
 *
 * COPYRIGHT 2005-2007 Koji Suzuki (suz@at.sakura.ne.jp)
 *   All Right Reserved 
 *
 */


/* 
               Tiny2313 (DIP)

           ~RESET 1   20 VCC 
 (1)ROW2  --- PD0 2   19 PB7 --- ROW1(16) SCK
 (2)COL4  -R- PD1 3   18 PB6 --- ROW3(14) MISO
 (3)COL3  -R- PA1 4   17 PB5 --- ROW4(13) MOSI
 (4)COL2  -R- PA0 5   16 PB4 -R- COL7(12)
 (5)ROW5  --- PD2 6   15 PB3 -R- COL6(11)
 (6)ROW6  --- PD3 7   14 PB2 -R- COL5(10)
 (7)COL1  -R- PD4 8   13 PB1 --- ROW7(9 )
 (8)ROW8  --- PD5 9   12 PB0 -R- COL8(15)
              GND 10  11 PD6 -+  
                              |
                            COMM
 VCC
  |
  C 10uF
  |
 GND

R: 51 for 3.3V  
  100 for 5.0V

        R: 51 @ 3.3V -> 30 mA 
           51 @ 5V   -> 65 mA (1/64 duty OK!) 
          100 @ 3.3V -> 16 mA 
          100 @ 5V   -> 33 mA 

BU5004R

    24 mm x 24 mm

         ROW2 1  o x x x x x o 16 ROW1
         COL4 2  o           o 15 COL8
         COL3 3  o           o 14 ROW3
         COL2 4  o           o 13 ROW4
         ROW5 5  o           o 12 COL7
         ROW6 6  o           o 11 COL6
         COL1 7  o           o 10 COL5
         ROW8 8  o           o  9 ROW7
                    (top view)

                             (A)
           +---------------+
           | | | | | | | | | ROW8
           +---------------+
           | | | | | | | | | ROW7
           +---------------+
                   :
           +---------------+
           | | | | | | | | | ROW1
           +---------------+
      COL   1 2 3 4 5 6 7 8
      (K)


fusebits note:
     set 8 MHz 
	   (low byte)  0x62 (default,4Mhz) -> 0x64 (8Mhz)
     set eeprom guard (at chip erase)
	   (high byte) 0xdf -> 0x9f

eeprom note:


serial memo: 

    PD6 -- RxD 56700 bps , 8n1

    protocol

    bit7-4 bit3-0   extra data num
    chan    cmd
     
             0          0  clear 
             1          8  upload data 
             2          1  save data to eeprom (extra data = page)

    chan : 1-15  select 
            0    broadcast

initialize 
   send 0 x 9

eeprom format 

page 0 : ( page size = 8)
	offset 0 : my chan
	offset 1 : disp mode 
	  0: direct mode (upload only)
	  1: direct text mode (upload only) (need undef USE_DIRECT_PROG_MODE)
	  2: eeprom mode 
	  3: eeprom text mode               (need undef USE_DIRECT_PROG_MODE)
	  4: movie mode                     (need define USE_DIRECT_PROG_MODE)
          5: movie with sequencer mode      (need define USE_DIRECT_PROG_MODE)
	offset 2 : disp time (frame count)  (note: frame rate is about 120 Hz)
	offset 3 : data length (bytes)
page 1 - 15 (data)

sequencer format :
    bit7-6   bit5-0
       0      frame#    show frame
       1      counter   load counter
       2      addr      if (--counter > 0) goto addr (indirect)
       3      tick      load  disp time

*/

   
#include <string.h>
#include <inttypes.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>

#define CPUCLOCK	8000000
#define BAUD		57600

//#define USE_DIRECT_PROG_MODE
#if !defined(USE_DIRECT_PROG_MODE)
#define USE_CHAR_MODE
#endif
#define USE_SERIAL

#undef bit_is_set
#undef bit_is_clear

#define bit_is_set(__data, __bit)		!!((__data) & (1<<(__bit)))
#define bit_is_clear(__data, __bit)		!((__data) & (1<<(__bit)))

#define bit_set(__data, __bit)		{__data |= (1<<__bit);}
#define bit_clear(__data, __bit)	{__data &= ~(1<<__bit);}

#include "buconf.h"

#ifdef USE_CHAR_MODE
#include "char_data.h"
#else
#if defined(USE_DIRECT_PROG_MODE) 
#include "frame_data.h"
#endif
#endif


uint8_t disp_data[8];
uint8_t load_data[8];

struct config {
	uint8_t my_chan;	/* page 0, offset 0 */
	uint8_t disp_mode;	/* page 0, offset 1 */
	uint8_t disp_tick;	/* page 0, offset 2 */
	uint8_t data_len;	/* page 0, offset 3 */
} conf;

#define DISP_MODE_DIRECT	0
#define DISP_MODE_CHAR		1
#define DISP_MODE_DIRECT_EEPROM	2
#define DISP_MODE_CHAR_EEPROM	3
#define DISP_MODE_DIRECT_PROG	4
#define DISP_MODE_DIRECT_PROG2	5

static inline void port_setup() {
	/* pull up anode */
	bit_set(COL1_PORT,COL1);
	bit_set(COL2_PORT,COL2);
	bit_set(COL3_PORT,COL3);
	bit_set(COL4_PORT,COL4);
	bit_set(COL5_PORT,COL5);
	bit_set(COL6_PORT,COL6);
	bit_set(COL7_PORT,COL7);
	bit_set(COL8_PORT,COL8);

	bit_set(PORTD,PD6);	/* pull up */
}

static int8_t cur_col,cur_row;

static inline void disp_off() {
	switch(cur_row) {
	case 0: bit_clear(COL1_DDR,COL1); break;
	case 1: bit_clear(COL2_DDR,COL2); break;
	case 2: bit_clear(COL3_DDR,COL3); break;
	case 3: bit_clear(COL4_DDR,COL4); break;
	case 4: bit_clear(COL5_DDR,COL5); break;
	case 5: bit_clear(COL6_DDR,COL6); break;
	case 6: bit_clear(COL7_DDR,COL7); break;
	case 7: bit_clear(COL8_DDR,COL8); break;
	}
	switch(cur_col) {
	case 0:  bit_clear(ROW1_DDR, ROW1); break;
	case 1:  bit_clear(ROW2_DDR, ROW2); break;
	case 2:  bit_clear(ROW3_DDR, ROW3); break;
	case 3:  bit_clear(ROW4_DDR, ROW4); break;
	case 4:  bit_clear(ROW5_DDR, ROW5); break;
	case 5:  bit_clear(ROW6_DDR, ROW6); break;
	case 6:  bit_clear(ROW7_DDR, ROW7); break;
	case 7:  bit_clear(ROW8_DDR, ROW8); break;
	}
}

static inline void disp_on(int8_t row,int8_t col) {
	switch(row) {
	case 0: bit_set(COL1_DDR,COL1); break;
	case 1: bit_set(COL2_DDR,COL2); break;
	case 2: bit_set(COL3_DDR,COL3); break;
	case 3: bit_set(COL4_DDR,COL4); break;
	case 4: bit_set(COL5_DDR,COL5); break;
	case 5: bit_set(COL6_DDR,COL6); break;
	case 6: bit_set(COL7_DDR,COL7); break;
	case 7: bit_set(COL8_DDR,COL8); break;
	}
	switch(col) {
	case 0:  bit_set(ROW1_DDR, ROW1); break;
	case 1:  bit_set(ROW2_DDR, ROW2); break;
	case 2:  bit_set(ROW3_DDR, ROW3); break;
	case 3:  bit_set(ROW4_DDR, ROW4); break;
	case 4:  bit_set(ROW5_DDR, ROW5); break;
	case 5:  bit_set(ROW6_DDR, ROW6); break;
	case 6:  bit_set(ROW7_DDR, ROW7); break;
	case 7:  bit_set(ROW8_DDR, ROW8); break;
	}
	cur_row = row;
	cur_col = col;
}

static inline void set_timeout(uint16_t t) {
	uint8_t v;
	if (250 > t) {
		v = 2; // source = clk/8
	} else {
	    t >>= 3;
	    if (250 > t) {
		    v = 3; // source = clk/64
	    } else {
	 	t >>= 2;
	        if (250 > t) {
		    v = 4; // source = clk/256
		} else {
	 		t >>= 2;
		    	v = 5; // source = clk/1024
	        	if (t >= 250) {
				t = 250;
			}
		}
	    }
	}
	TCCR0B = 0; /* stop */
	bit_set(TIFR, TOV0);
        TCNT0 = 0 - (uint8_t)t;
	TCCR0B = v;
}

static inline int8_t check_timeout() {
	return bit_is_set(TIFR, TOV0);
}

#ifdef USE_SERIAL
#define ICP_IBUF_SIZE	8
volatile uint8_t icp_ibuf_r;
volatile uint8_t icp_ibuf_w;
uint8_t icp_ibuf[ICP_IBUF_SIZE];

static inline int8_t icp_ibuf_empty() {
	return (icp_ibuf_r == icp_ibuf_w);
}

static inline int8_t icp_ibuf_full() {
	return (((icp_ibuf_w + 1)&(ICP_IBUF_SIZE -1)) == icp_ibuf_r);
}

static inline int8_t icp_can_getc() {
	return !icp_ibuf_empty();
}

static inline int8_t icp_getc() {
	int8_t c;

	while (icp_ibuf_empty())
		;
	cli(); // disable interrupt
	c = icp_ibuf[icp_ibuf_r];
	icp_ibuf_r = (icp_ibuf_r + 1) & (ICP_IBUF_SIZE -1);
	sei(); // enable interrupt
	return c;
}

int8_t icp_state;
uint8_t icp_data;

static inline void icp_setup() {
	/* setup PD6  change */
	TCCR1B = (1<<ICNC1) | 1 ; /* capture , 1/1 CLK select */
	TIMSK |= (1<<OCIE1A) | (1<<ICIE1);
	icp_state = 0;
}

/* bind start bit */
SIGNAL(SIG_TIMER1_CAPT) {
	/* 
	 *   4 + 4 + about 15 clock             4 + about 15 clock
	 * | <=> NOW                             <=>READ PD6
	 * |      |                             |   |
	 * +---------------------------+-----------------------------+
	 * <----- CPUCLOCK/BAUD x 1.5 -------------->
	 *
	 */
	TCCR1B = 0;
	TCNT1 = 0;
	OCR1A = CPUCLOCK * 3/2 / BAUD - 4 - 4 - 15 - 4 - 15; /* 1.5 tick */
	TCCR1B = (1<<WGM12) | 1 ; /* compare OCR1A  , 1/1 CLK select */
	icp_state = 1;	/* next is capture bit 0 */
}

/* processing each bit */
SIGNAL(SIG_TIMER1_COMPA) {
	if (icp_state <= 8) {
		icp_data >>= 1;
		if (bit_is_set(PIND,PD6))
			icp_data |= 0x80;
		icp_state++;
		OCR1A = CPUCLOCK / BAUD; /* 1 tick (just) */
	} else {
		if (bit_is_set(PIND,PD6) && !icp_ibuf_full()) {
			icp_ibuf[icp_ibuf_w] = icp_data;
			icp_ibuf_w = (icp_ibuf_w + 1) & (ICP_IBUF_SIZE -1);
		}
		icp_setup();
	}
}

static int8_t recv_count;
static int8_t recv_state;

#define CMD_CLEAR	0
#define CMD_UPLOAD	1
#define CMD_SAVE	2

static void recv_proc(uint8_t c) {
	uint8_t cmd = c & 0xf;

	if (recv_count == 0) {
		recv_state = 0;
		if ((c & 0xf0) == 0 || (c & 0xf0) == conf.my_chan) {
			recv_state++;
		}
		if (cmd == CMD_CLEAR) {
			/* clear */
			if (recv_state) {
				memset(disp_data,0, 8);
				memset(load_data,0, 8);
			}
		} else if (cmd == CMD_UPLOAD) {
			/* download full data */
			recv_count = 8;
			if (recv_state) recv_state++;
		} else if (cmd == CMD_SAVE) {
			recv_count++;
			if (recv_state) recv_state = 3;
		}
	} else {
		recv_count --;
		if (recv_state == 2) {
			load_data[7 - recv_count] = c;
		} else if (recv_state == 3) { /* SAVE DATA to EEPROM */
			eeprom_write_block(load_data, 
				(void *)((int16_t)(cmd) << 3),8);
			if (cmd == 0) {
				memcpy(&conf, load_data, sizeof(conf));
			}
			if (conf.disp_mode == DISP_MODE_DIRECT) {
				memcpy(disp_data, load_data, 8);
			}
		}
	}
}

#endif

uint8_t frame_count;
uint8_t disp_pos;
uint8_t disp_pos2;
uint16_t disp_idx;

static void run_proc() {
#ifdef USE_DIRECT_PROG_MODE
	int8_t i;
#endif
	uint16_t idx;
	frame_count ++;
	if (frame_count >= conf.disp_tick) {
		frame_count = 0;
		switch (conf.disp_mode) {
		case DISP_MODE_DIRECT_EEPROM:
			memcpy(disp_data, disp_data+1, 7);
			idx = disp_pos + 8;
			disp_data[7] = eeprom_read_byte((void *)idx);
			disp_pos++;
			if (disp_pos >= conf.data_len) disp_pos = 0;
			break;
#ifdef USE_DIRECT_PROG_MODE
		case DISP_MODE_DIRECT_PROG:
			for (i=0; i<8; i++) {
				disp_data[i] = __LPM(prog_data + disp_idx);
				disp_idx++;
			}
			if ((disp_idx>>8) >= conf.data_len) disp_idx = 0;
			break;
		case DISP_MODE_DIRECT_PROG2:
			for (i=0; i<3; i++) {
				static int8_t counter;
				idx = disp_pos + 8;
				disp_pos2 = eeprom_read_byte((void *)idx);
				switch (disp_pos2 & 0xc0) {
				case 0x40: /* load counter */
					counter = disp_pos2 & 0x3f;
					goto next;
				case 0x80: /* if (-- counter <= 0) next
					      else goto value */
					counter--;
					if (counter > 0) {
						i = disp_pos2;
						i <<= 2;
						i >>= 2;
						disp_pos += i;
						break;
					}
					goto next;
				case 0xc0: /* load disp tick */
					conf.disp_tick = disp_pos2 & 0x3f;
				case 0x00: /* next and accept */
				next:
					disp_pos++;
					break;
				}
				if (disp_pos >= conf.data_len)
					disp_pos = 0;
				if (!(disp_pos2 & 0xc0)) break;
			}
			idx = disp_pos2;
			idx <<= 3;
			for (i=0; i<8; i++) {
				disp_data[i] = __LPM(prog_data + idx);
				idx++;
			}
			break;
#endif
#ifdef USE_CHAR_MODE
		case DISP_MODE_CHAR:
		case DISP_MODE_CHAR_EEPROM:
			memcpy(disp_data, disp_data+1, 7);
			if (disp_pos2 == 0) {
				if (conf.disp_mode == DISP_MODE_CHAR) {
					disp_idx = load_data[disp_pos];
				} else {
					idx = disp_pos + 8;
					disp_idx = eeprom_read_byte((void *)idx);
				}
				disp_idx -= 0x20;
				disp_idx += (disp_idx << 2); /* *= 5 */
				disp_pos++;
				if (disp_pos >= conf.data_len) disp_pos = 0;
				disp_data[7] = 0;
			} else {
				disp_data[7] = __LPM(prog_data + disp_idx);
				disp_idx++;
			}
			disp_pos2++;
			if (disp_pos2 > 5) {
				disp_pos2 = 0;
			}
			break;
#endif
		}
	}
}

int main() {
	int8_t i, j;
	uint8_t bit;

	/* set system_clock pre scaler 1/8 => 1/1 */
	CLKPR = (1<<CLKPCE);
	CLKPR = 0;

	port_setup();

#ifdef USE_SERIAL
	icp_setup();
#endif
	sei();

	eeprom_read_block(&conf, (const void *)(0),sizeof(conf));

	i = j = 0;
	bit = 1;
	set_timeout(100);
	for (;;) {
#ifdef USE_SERIAL
		if (!icp_ibuf_empty()) {
			recv_proc(icp_getc());
		}
#endif
		if (check_timeout()) {
			disp_off();
			set_timeout(100);
			if (disp_data[i] & bit) disp_on(i,j);
			bit <<= 1;
			j++;
			if (j == 8) {
				j = 0;
				bit = 1;
				i++;
				if (i == 8) {
					i = 0;
					/* 110 - 130 Hz */
					run_proc();
				}
			}
		}
	}
	return 0;
}
